<?php
// Simple diagnostic script to check server configuration
echo "<h1>Server Diagnostic</h1>";

// 1. PHP Version
echo "<h2>1. PHP Version</h2>";
echo "PHP Version: " . phpversion() . "<br>";
echo "Required: 8.0 or higher<br>";
if (version_compare(phpversion(), '8.0.0', '>=')) {
    echo "✅ PHP version is compatible<br>";
} else {
    echo "❌ PHP version is too old. Need 8.0+<br>";
}

// 2. Required Extensions
echo "<h2>2. Required PHP Extensions</h2>";
$required = ['pdo', 'pdo_mysql', 'mysqli', 'mbstring', 'json'];
foreach ($required as $ext) {
    if (extension_loaded($ext)) {
        echo "✅ {$ext} is loaded<br>";
    } else {
        echo "❌ {$ext} is NOT loaded<br>";
    }
}

// 3. File Paths
echo "<h2>3. File Paths</h2>";
echo "Current file: " . __FILE__ . "<br>";
echo "Document root: " . $_SERVER['DOCUMENT_ROOT'] . "<br>";
echo "Script filename: " . $_SERVER['SCRIPT_FILENAME'] . "<br>";

// 4. Check if config.php exists
echo "<h2>4. Configuration Files</h2>";
$config_path = __DIR__ . '/config.php';
if (file_exists($config_path)) {
    echo "✅ config.php exists at: {$config_path}<br>";
    echo "File permissions: " . substr(sprintf('%o', fileperms($config_path)), -4) . "<br>";
    
    // Try to include it
    try {
        require_once $config_path;
        echo "✅ config.php loaded successfully<br>";
        
        // Check if constants are defined
        if (defined('DB_HOST')) {
            echo "✅ DB_HOST defined: " . DB_HOST . "<br>";
        } else {
            echo "❌ DB_HOST not defined<br>";
        }
        
        if (defined('DB_NAME')) {
            echo "✅ DB_NAME defined: " . DB_NAME . "<br>";
        } else {
            echo "❌ DB_NAME not defined<br>";
        }
        
        if (defined('DB_USER')) {
            echo "✅ DB_USER defined: " . DB_USER . "<br>";
        } else {
            echo "❌ DB_USER not defined<br>";
        }
        
    } catch (Exception $e) {
        echo "❌ Error loading config.php: " . $e->getMessage() . "<br>";
    }
} else {
    echo "❌ config.php NOT found at: {$config_path}<br>";
}

// 5. Check if includes directory exists
echo "<h2>5. Includes Directory</h2>";
$includes_path = __DIR__ . '/includes';
if (is_dir($includes_path)) {
    echo "✅ includes/ directory exists<br>";
    $files = scandir($includes_path);
    echo "Files in includes/: " . implode(', ', array_diff($files, ['.', '..'])) . "<br>";
} else {
    echo "❌ includes/ directory NOT found<br>";
}

// 6. Database Connection Test
echo "<h2>6. Database Connection Test</h2>";
if (defined('DB_HOST') && defined('DB_NAME') && defined('DB_USER') && defined('DB_PASS')) {
    try {
        $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4";
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ];
        $pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
        echo "✅ Database connection successful!<br>";
        
        // Test query
        $stmt = $pdo->query("SELECT DATABASE() as db");
        $result = $stmt->fetch();
        echo "✅ Connected to database: " . $result['db'] . "<br>";
        
        // Check if app_settings table exists
        $stmt = $pdo->query("SHOW TABLES LIKE 'app_settings'");
        if ($stmt->rowCount() > 0) {
            echo "✅ app_settings table exists<br>";
            
            // Count rows
            $stmt = $pdo->query("SELECT COUNT(*) as count FROM app_settings");
            $result = $stmt->fetch();
            echo "✅ app_settings has {$result['count']} rows<br>";
        } else {
            echo "❌ app_settings table does NOT exist<br>";
        }
        
    } catch (PDOException $e) {
        echo "❌ Database connection failed: " . $e->getMessage() . "<br>";
    }
} else {
    echo "❌ Database constants not defined<br>";
}

// 7. Error Reporting
echo "<h2>7. PHP Configuration</h2>";
echo "display_errors: " . ini_get('display_errors') . "<br>";
echo "error_reporting: " . ini_get('error_reporting') . "<br>";
echo "log_errors: " . ini_get('log_errors') . "<br>";
echo "error_log: " . ini_get('error_log') . "<br>";

// 8. Memory and Execution
echo "<h2>8. Resource Limits</h2>";
echo "memory_limit: " . ini_get('memory_limit') . "<br>";
echo "max_execution_time: " . ini_get('max_execution_time') . "<br>";
echo "upload_max_filesize: " . ini_get('upload_max_filesize') . "<br>";

echo "<hr>";
echo "<h2>✅ Diagnostic Complete</h2>";
echo "<p>If you see this page, PHP is working. Check the items marked with ❌ above.</p>";
?>
