<?php
// File: test_api.php
// Simple test script to verify API setup

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html>
<head>
    <title>ensi.lk API Test</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 800px;
            margin: 50px auto;
            padding: 20px;
            background: #f5f5f5;
        }
        .test-box {
            background: white;
            padding: 20px;
            margin: 20px 0;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .success { color: #00B894; }
        .error { color: #FF7675; }
        .info { color: #6C63FF; }
        h1 { color: #6C63FF; }
        h2 { color: #2D3436; margin-top: 0; }
        pre {
            background: #f8f9fa;
            padding: 10px;
            border-radius: 5px;
            overflow-x: auto;
        }
    </style>
</head>
<body>
    <h1>🎓 ensi.lk API Test Suite</h1>
    
    <?php
    require_once 'config.php';
    
    echo '<div class="test-box">';
    echo '<h2>1. Database Connection Test</h2>';
    
    $conn = getDBConnection();
    if ($conn) {
        echo '<p class="success">✓ Database connection successful!</p>';
        
        // Test tables
        echo '<h3>Checking Tables:</h3>';
        $tables = ['users', 'verification_codes', 'user_sessions'];
        foreach ($tables as $table) {
            try {
                $stmt = $conn->query("SELECT COUNT(*) as count FROM $table");
                $result = $stmt->fetch();
                echo '<p class="success">✓ Table <strong>' . $table . '</strong> exists (' . $result['count'] . ' records)</p>';
            } catch (Exception $e) {
                echo '<p class="error">✗ Table <strong>' . $table . '</strong> not found</p>';
            }
        }
    } else {
        echo '<p class="error">✗ Database connection failed</p>';
        echo '<p>Check config.php credentials</p>';
    }
    echo '</div>';
    
    // Test PHPMailer
    echo '<div class="test-box">';
    echo '<h2>2. PHPMailer Check</h2>';
    
    $phpmailerFiles = ['PHPMailer.php', 'SMTP.php', 'Exception.php'];
    $allFilesExist = true;
    
    foreach ($phpmailerFiles as $file) {
        if (file_exists($file)) {
            echo '<p class="success">✓ ' . $file . ' found</p>';
        } else {
            echo '<p class="error">✗ ' . $file . ' not found</p>';
            $allFilesExist = false;
        }
    }
    
    if ($allFilesExist) {
        echo '<p class="info">All PHPMailer files are present!</p>';
    } else {
        echo '<p class="error">Missing PHPMailer files. Please download them.</p>';
    }
    echo '</div>';
    
    // Test API endpoints
    echo '<div class="test-box">';
    echo '<h2>3. API Endpoints</h2>';
    
    $endpoints = [
        'register.php',
        'verify_email.php',
        'resend_code.php',
        'complete_registration.php',
        'login.php',
        'request_password_reset.php',
        'reset_password.php',
        'delete_account.php',
        'verify_delete.php',
    ];
    
    foreach ($endpoints as $endpoint) {
        if (file_exists($endpoint)) {
            echo '<p class="success">✓ ' . $endpoint . '</p>';
        } else {
            echo '<p class="error">✗ ' . $endpoint . ' missing</p>';
        }
    }
    echo '</div>';
    
    // Test PHP configuration
    echo '<div class="test-box">';
    echo '<h2>4. PHP Configuration</h2>';
    echo '<p><strong>PHP Version:</strong> ' . phpversion() . '</p>';
    echo '<p><strong>Timezone:</strong> ' . date_default_timezone_get() . '</p>';
    echo '<p><strong>Max Upload Size:</strong> ' . ini_get('upload_max_filesize') . '</p>';
    echo '<p><strong>Max POST Size:</strong> ' . ini_get('post_max_size') . '</p>';
    echo '<p><strong>Memory Limit:</strong> ' . ini_get('memory_limit') . '</p>';
    
    // Check if PDO MySQL is available
    if (extension_loaded('pdo_mysql')) {
        echo '<p class="success">✓ PDO MySQL extension loaded</p>';
    } else {
        echo '<p class="error">✗ PDO MySQL extension not loaded</p>';
    }
    
    // Check if OpenSSL is available (needed for SMTP)
    if (extension_loaded('openssl')) {
        echo '<p class="success">✓ OpenSSL extension loaded (needed for SMTP)</p>';
    } else {
        echo '<p class="error">✗ OpenSSL extension not loaded</p>';
    }
    echo '</div>';
    
    // Test sample API call
    echo '<div class="test-box">';
    echo '<h2>5. Test Sample Response</h2>';
    echo '<p class="info">Testing JSON response format:</p>';
    
    $testResponse = [
        'success' => true,
        'message' => 'API is working correctly!',
        'data' => [
            'version' => '1.0.0',
            'timestamp' => date('Y-m-d H:i:s')
        ]
    ];
    
    echo '<pre>' . json_encode($testResponse, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE) . '</pre>';
    echo '</div>';
    
    // Logs directory check
    echo '<div class="test-box">';
    echo '<h2>6. Logs Directory</h2>';
    
    if (file_exists('logs') && is_dir('logs')) {
        if (is_writable('logs')) {
            echo '<p class="success">✓ Logs directory exists and is writable</p>';
            
            // Test log writing
            $testMessage = "Test log entry at " . date('Y-m-d H:i:s');
            logActivity($testMessage);
            
            if (file_exists('logs/activity.log')) {
                echo '<p class="success">✓ Successfully wrote to activity.log</p>';
            }
        } else {
            echo '<p class="error">✗ Logs directory exists but is not writable</p>';
            echo '<p>Run: chmod 755 logs</p>';
        }
    } else {
        echo '<p class="error">✗ Logs directory does not exist</p>';
        echo '<p>Run: mkdir logs && chmod 755 logs</p>';
    }
    echo '</div>';
    
    ?>
    
    <div class="test-box">
        <h2>Summary</h2>
        <p>If all tests passed, your API is ready to use! 🎉</p>
        <p>If you see any errors, please fix them before proceeding.</p>
        <p><strong>Next Steps:</strong></p>
        <ol>
            <li>Delete or restrict access to this test_api.php file</li>
            <li>Configure your email settings in send_email.php</li>
            <li>Test the registration flow from the Flutter app</li>
        </ol>
    </div>
</body>
</html>
