<?php
/**
 * Test subscription pages with authentication
 */
require_once 'config.php';

// Start session
if (session_status() === PHP_SESSION_NONE) {
    session_name(ADMIN_SESSION_NAME);
    session_start();
}

echo "<h2>Subscription Pages Test (Authenticated)</h2>";
echo "<hr>";

// Check if logged in
if (!isset($_SESSION['admin_id'])) {
    echo "<p style='color: red;'>❌ You are not logged in!</p>";
    echo "<p><a href='login.php'>Click here to login first</a></p>";
    exit;
}

echo "<p>✅ Logged in as Admin ID: {$_SESSION['admin_id']}</p>";
echo "<hr>";

$pdo = getDBConnection();

// Test 1: Get admin info
echo "<h3>Test 1: Get Current Admin</h3>";
try {
    $admin = getCurrentAdmin();
    if ($admin) {
        echo "✅ Admin loaded: " . htmlspecialchars($admin['username']) . " (" . $admin['email'] . ")<br>";
        echo "Role: " . $admin['role'] . "<br>";
    } else {
        echo "❌ Failed to load admin data<br>";
    }
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "<br>";
}

echo "<hr>";

// Test 2: Get subscriptions
echo "<h3>Test 2: Load Subscriptions Query</h3>";
try {
    $query = "
        SELECT 
            s.id as subscription_id,
            s.status,
            s.payment_method,
            s.amount_paid,
            s.start_date,
            s.end_date,
            s.created_at,
            DATEDIFF(s.end_date, NOW()) as days_remaining,
            u.id as user_id,
            u.email,
            u.user_name,
            COALESCE(u.full_name, u.user_name, u.email) as display_name,
            p.package_name,
            p.duration_months
        FROM user_subscriptions s
        JOIN users u ON s.user_id = u.id
        JOIN subscription_packages p ON s.package_id = p.id
        ORDER BY s.created_at DESC
        LIMIT 10
    ";
    
    $stmt = $pdo->query($query);
    $subscriptions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo "✅ Query executed successfully<br>";
    echo "Found " . count($subscriptions) . " subscriptions<br>";
    
    if (count($subscriptions) > 0) {
        echo "<pre>";
        print_r($subscriptions[0]); // Show first record
        echo "</pre>";
    }
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "<br>";
}

echo "<hr>";

// Test 3: Get statistics
echo "<h3>Test 3: Load Statistics</h3>";
try {
    $statsQuery = "
        SELECT 
            COUNT(CASE WHEN status = 'active' THEN 1 END) as active_count,
            COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_count,
            COUNT(CASE WHEN status = 'expired' THEN 1 END) as expired_count,
            SUM(CASE WHEN status = 'active' THEN amount_paid ELSE 0 END) as active_revenue,
            SUM(CASE WHEN status = 'pending' THEN amount_paid ELSE 0 END) as pending_revenue
        FROM user_subscriptions
    ";
    $stats = $pdo->query($statsQuery)->fetch(PDO::FETCH_ASSOC);
    
    echo "✅ Statistics loaded:<br>";
    echo "Active: " . $stats['active_count'] . "<br>";
    echo "Pending: " . $stats['pending_count'] . "<br>";
    echo "Expired: " . $stats['expired_count'] . "<br>";
    echo "Revenue: Rs " . number_format($stats['active_revenue'] + $stats['pending_revenue'], 2) . "<br>";
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "<br>";
}

echo "<hr>";

// Test 4: Get payment requests
echo "<h3>Test 4: Load Payment Requests</h3>";
try {
    $query = "
        SELECT 
            pr.id as request_id,
            pr.amount,
            pr.payment_method,
            pr.payment_screenshot,
            pr.depositor_name,
            pr.deposit_date,
            pr.bank_reference,
            pr.created_at,
            u.id as user_id,
            u.email,
            u.user_name,
            COALESCE(u.full_name, u.user_name, u.email) as display_name,
            p.package_name,
            p.duration_months,
            s.id as subscription_id
        FROM payment_requests pr
        JOIN users u ON pr.user_id = u.id
        JOIN subscription_packages p ON pr.package_id = p.id
        JOIN user_subscriptions s ON pr.subscription_id = s.id
        WHERE pr.status = 'pending'
        ORDER BY pr.created_at ASC
        LIMIT 10
    ";
    
    $stmt = $pdo->query($query);
    $requests = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo "✅ Query executed successfully<br>";
    echo "Found " . count($requests) . " pending payment requests<br>";
    
    if (count($requests) > 0) {
        echo "<pre>";
        print_r($requests[0]); // Show first record
        echo "</pre>";
    }
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "<br>";
}

echo "<hr>";
echo "<h3>Conclusion:</h3>";
echo "<p>If all tests passed above, the subscription pages should work!</p>";
echo "<p><a href='subscriptions.php'>Click here to test Subscriptions page</a></p>";
echo "<p><a href='payment_requests.php'>Click here to test Payment Requests page</a></p>";
?>
